jQuery(document).ready(function ($) {

    var editor       = null;
    var $target      = null;
    var originalHTML = '';
    var isDirty      = false;
    var isEditing    = false;

    var $bar         = $('#rfhe-bar');
    var $editBtn     = $('#rfhe-edit-btn');
    var $editActions = $('#rfhe-edit-actions');
    var $saveBtn     = $('#rfhe-save-btn');
    var $cancelBtn   = $('#rfhe-cancel-btn');
    var $status      = $('#rfhe-status');
    var $pickBtn     = $('#rfhe-pick-btn');

    var cmTheme = (rfheData.theme === 'light') ? 'eclipse' : 'dracula';

    // Mark body so CSS can safely add toolbar offset without affecting all visitors.
    $('body').addClass('rfhe-active');

    // ── Format toolbar HTML (built dynamically to include custom snippets) ─────

    function buildFmtHtml() {
        var t   = rfheData.theme;
        var html =
            '<div id="rfhe-fmt" class="rfhe-theme-' + t + '">' +
                '<div class="rfhe-fmt-group">' +
                    '<button data-tag="strong" title="Bold (Ctrl+B)"><b>B</b></button>' +
                    '<button data-tag="em"     title="Italic (Ctrl+I)"><i>I</i></button>' +
                    '<button data-tag="s"      title="Strikethrough"><s>S</s></button>' +
                    '<button data-tag="mark"   title="Highlight">M</button>' +
                    '<button data-tag="code"   title="Inline code">&lt;/&gt;</button>' +
                '</div>' +
                '<div class="rfhe-fmt-sep"></div>' +
                '<div class="rfhe-fmt-group">' +
                    '<button data-wrap="h2" title="Heading 2">H2</button>' +
                    '<button data-wrap="h3" title="Heading 3">H3</button>' +
                    '<button data-wrap="p"  title="Paragraph">P</button>' +
                    '<button data-wrap="blockquote" title="Blockquote">&#10077;</button>' +
                '</div>' +
                '<div class="rfhe-fmt-sep"></div>' +
                '<div class="rfhe-fmt-group">' +
                    '<button data-action="ul"   title="Unordered list">&#8801;</button>' +
                    '<button data-action="ol"   title="Ordered list">1.</button>' +
                    '<button data-action="link" title="Insert link (Ctrl+K)">&#128279;</button>' +
                    '<button data-action="img"  title="Insert image">&#128444;</button>' +
                '</div>';

        // Custom snippets
        var snippets = rfheData.snippets || [];
        if (snippets.length) {
            html += '<div class="rfhe-fmt-sep"></div>';
            html += '<div class="rfhe-fmt-group rfhe-fmt-snippets">';
            snippets.forEach(function (s, i) {
                html += '<button data-snippet="' + i + '" title="' + escAttr(s.label) + '" class="rfhe-snippet-btn">'
                      + escHtml(s.label)
                      + '</button>';
            });
            html += '</div>';
        }

        html +=
                '<div class="rfhe-fmt-sep"></div>' +
                '<div class="rfhe-fmt-hints">' +
                    '<span>Ctrl+B</span><span>Ctrl+I</span><span>Ctrl+K</span><span>Ctrl+S</span>' +
                '</div>' +
            '</div>';

        return html;
    }

    // ── Element Picker ────────────────────────────────────────────────────────

    var isPicking    = false;
    var $highlighted = null;

    $pickBtn.on('click', function () {
        if (isPicking) { stopPicker(); return; }
        startPicker();
    });

    function startPicker() {
        isPicking = true;
        $pickBtn.text('✕ Cancel pick').addClass('rfhe-picking');
        $('body').addClass('rfhe-pick-mode');
        setStatus('Click on any content block to get its selector', 'info');

        $(document).on('mouseover.rfhe-pick', function (e) {
            if ($(e.target).closest('#rfhe-bar, #rfhe-debug-panel').length) return;
            if ($highlighted) $highlighted.removeClass('rfhe-pick-hover');
            $highlighted = $(e.target);
            $highlighted.addClass('rfhe-pick-hover');
        });

        $(document).on('click.rfhe-pick', function (e) {
            if ($(e.target).closest('#rfhe-bar').length) return;
            e.preventDefault();
            e.stopPropagation();
            var el  = e.target;
            var sel = buildSelector(el);
            stopPicker();
            showPickerResult(sel, el);
        });
    }

    function stopPicker() {
        isPicking = false;
        $pickBtn.text('🎯 Pick element').removeClass('rfhe-picking');
        $('body').removeClass('rfhe-pick-mode');
        if ($highlighted) { $highlighted.removeClass('rfhe-pick-hover'); $highlighted = null; }
        $(document).off('mouseover.rfhe-pick click.rfhe-pick');
    }

    function buildSelector(el) {
        if (el.id) return '#' + el.id;
        var classes = Array.from(el.classList).filter(function (c) {
            return !c.startsWith('rfhe-') && document.querySelectorAll('.' + c).length === 1;
        });
        if (classes.length) return '.' + classes[0];
        if (el.classList.length) {
            var cls = Array.from(el.classList).filter(function (c) { return !c.startsWith('rfhe-'); });
            if (cls.length) return el.tagName.toLowerCase() + '.' + cls[0];
        }
        return el.tagName.toLowerCase();
    }

    function getAncestors(el) {
        var chain = [], cur = el;
        while (cur && cur.tagName && cur.tagName !== 'BODY' && cur.tagName !== 'HTML') {
            chain.push(cur);
            cur = cur.parentElement;
        }
        return chain;
    }

    function showPickerResult(sel, el) {
        $('#rfhe-pick-result').remove();

        var ancestors  = getAncestors(el);
        var currentIdx = 0;

        function renderAncestorTag(node) {
            var t = node.tagName.toLowerCase();
            if (node.id) return t + '#' + node.id;
            var cls = typeof node.className === 'string'
                ? node.className.trim().replace(/\s+/g, '.')
                    .replace(/rfhe-[^.]*/g, '').replace(/\.+/g, '.').replace(/^\.|\.$/g, '')
                : '';
            return t + (cls ? '.' + cls.split('.')[0] : '');
        }

        function currentSel() { return buildSelector(ancestors[currentIdx]); }

        function rebuildBreadcrumbs() {
            var html = '';
            ancestors.forEach(function (node, i) {
                var label = renderAncestorTag(node);
                html += '<span class="rfhe-crumb' + (i === currentIdx ? ' rfhe-crumb-active' : '') + '" data-idx="' + i + '">' + escHtml(label) + '</span>';
                if (i < ancestors.length - 1) html += '<span class="rfhe-crumb-sep">›</span>';
            });
            return html;
        }

        function update() {
            var s = currentSel();
            $('#rfhe-pick-breadcrumbs').html(rebuildBreadcrumbs());
            $('#rfhe-pick-sel').text(s);
            $('#rfhe-pick-breadcrumbs .rfhe-crumb').on('click', function () {
                currentIdx = parseInt($(this).data('idx'));
                update();
            });
        }

        var html = '<div id="rfhe-pick-result">'
            + '<button id="rfhe-pick-close">✕</button>'
            + '<div class="rfhe-pick-title">🎯 Element found — click a parent to select it</div>'
            + '<div id="rfhe-pick-breadcrumbs"></div>'
            + '<div class="rfhe-pick-row">'
            + '  <span class="rfhe-pick-label">Selector:</span>'
            + '  <code id="rfhe-pick-sel"></code>'
            + '  <button id="rfhe-pick-copy">Copy</button>'
            + '  <a href="' + rfheData.settings_url + '" target="_blank" id="rfhe-pick-settings">Save in Settings →</a>'
            + '</div>'
            + '<button id="rfhe-pick-use">✓ Use this selector now (current session only)</button>'
            + '</div>';

        $('body').append(html);
        update();

        $('#rfhe-pick-copy').on('click', function () {
            navigator.clipboard.writeText(currentSel()).then(function () {
                $('#rfhe-pick-copy').text('Copied!');
                setTimeout(function () { $('#rfhe-pick-copy').text('Copy'); }, 2000);
            });
        });

        $('#rfhe-pick-use').on('click', function () {
            rfheData.selector = currentSel();
            setStatus('Selector: ' + currentSel(), 'ok');
            $('#rfhe-pick-result').remove();
        });

        $('#rfhe-pick-close').on('click', function () {
            $('#rfhe-pick-result').remove();
        });
    }

    function escHtml(s) {
        return String(s)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;');
    }

    function escAttr(s) {
        return String(s).replace(/"/g, '&quot;').replace(/'/g, '&#39;');
    }

    // ── Find content container ────────────────────────────────────────────────

    function findTarget() {
        var sel = rfheData.selector || '.entry-content';
        var $el = $(sel);
        if ($el.length) return $el.first();
        var fallbacks = ['.entry-content', '.post-content', '.wp-block-post-content', 'article .content', 'main article'];
        for (var i = 0; i < fallbacks.length; i++) {
            $el = $(fallbacks[i]);
            if ($el.length) return $el.first();
        }
        return null;
    }

    // ── Enter edit mode ───────────────────────────────────────────────────────

    $editBtn.on('click', function () {
        $target = findTarget();
        if (!$target) {
            setStatus('Content block not found. Use 🎯 Pick element to find the right selector.', 'error');
            showEditActions();
            return;
        }

        setStatus('Loading...', 'info');
        showEditActions();

        $.post(rfheData.ajax_url, {
            action:  'rfhe_get_content',
            post_id: rfheData.post_id,
            nonce:   rfheData.nonce,
        }, function (res) {
            if (!res.success) { setStatus('Load error', 'error'); return; }

            originalHTML = $target.html();
            $target.addClass('rfhe-editing');
            $target.html(
                '<div id="rfhe-editor-box">' +
                    buildFmtHtml() +
                    '<div id="rfhe-cm-wrap"><textarea id="rfhe-textarea"></textarea></div>' +
                '</div>'
            );

            editor = CodeMirror.fromTextArea(
                document.getElementById('rfhe-textarea'), {
                    mode:          'htmlmixed',
                    theme:         cmTheme,
                    lineNumbers:   true,
                    lineWrapping:  true,
                    autoCloseTags: true,
                    matchBrackets: true,
                    indentUnit:    2,
                    tabSize:       2,
                    autofocus:     true,
                    extraKeys: {
                        'Ctrl-S': doSave,  'Cmd-S': doSave,
                        'Escape': doCancel,
                        'Ctrl-B': function (cm) { wrapSel(cm, 'strong'); },
                        'Cmd-B':  function (cm) { wrapSel(cm, 'strong'); },
                        'Ctrl-I': function (cm) { wrapSel(cm, 'em'); },
                        'Cmd-I':  function (cm) { wrapSel(cm, 'em'); },
                        'Ctrl-K': function (cm) { insertLink(cm); },
                        'Cmd-K':  function (cm) { insertLink(cm); },
                    }
                }
            );

            editor.setValue(res.data.content);
            editor.clearHistory();
            editor.refresh();
            editor.on('change', function () { isDirty = true; setStatus('Modified', 'warn'); });

            bindFmtBar();

            isEditing = true;
            $bar.addClass('rfhe-bar-editing').removeClass('rfhe-bar-preview');
            setStatus('Ready to edit', 'ok');

        }).fail(function () { setStatus('Network error', 'error'); });
    });

    // ── Format bar bindings ───────────────────────────────────────────────────

    function bindFmtBar() {
        $target.on('mousedown.rfhe', '#rfhe-fmt [data-tag]', function (e) {
            e.preventDefault();
            if (!editor) return;
            wrapSel(editor, $(this).data('tag'));
            editor.focus();
        });
        $target.on('mousedown.rfhe', '#rfhe-fmt [data-wrap]', function (e) {
            e.preventDefault();
            if (!editor) return;
            wrapBlock(editor, $(this).data('wrap'));
            editor.focus();
        });
        $target.on('mousedown.rfhe', '#rfhe-fmt [data-action]', function (e) {
            e.preventDefault();
            if (!editor) return;
            var a = $(this).data('action');
            if (a === 'link') insertLink(editor);
            if (a === 'img')  insertImg(editor);
            if (a === 'ul')   insertList(editor, 'ul');
            if (a === 'ol')   insertList(editor, 'ol');
            editor.focus();
        });

        // Custom snippet buttons
        $target.on('mousedown.rfhe', '#rfhe-fmt [data-snippet]', function (e) {
            e.preventDefault();
            if (!editor) return;
            var idx     = parseInt($(this).data('snippet'), 10);
            var snippet = (rfheData.snippets || [])[idx];
            if (snippet) insertSnippet(editor, snippet.code);
            editor.focus();
        });
    }

    // ── Format helpers ────────────────────────────────────────────────────────

    function wrapSel(cm, tag) {
        var sel  = cm.getSelection();
        var open = '<' + tag + '>', close = '</' + tag + '>';
        if (!sel) {
            var cur = cm.getCursor();
            cm.replaceRange(open + close, cur);
            cm.setCursor({ line: cur.line, ch: cur.ch + open.length });
            return;
        }
        cm.replaceSelection(
            (sel.startsWith(open) && sel.endsWith(close))
                ? sel.slice(open.length, -close.length)
                : open + sel + close
        );
    }

    function wrapBlock(cm, tag) {
        var sel = cm.getSelection().trim();
        if (!sel) {
            var line = cm.getCursor().line;
            var txt  = stripOuterTag(cm.getLine(line).trim());
            cm.setSelection({ line: line, ch: 0 }, { line: line, ch: cm.getLine(line).length });
            cm.replaceSelection('<' + tag + '>' + txt + '</' + tag + '>');
        } else {
            cm.replaceSelection('<' + tag + '>' + stripOuterTag(sel) + '</' + tag + '>');
        }
    }

    function stripOuterTag(s) {
        return s.replace(/^<[a-z0-9]+[^>]*>([\s\S]*)<\/[a-z0-9]+>$/i, '$1').trim();
    }

    function insertLink(cm) {
        var sel = cm.getSelection();
        var url = prompt('URL:', 'https://');
        if (!url) return;
        var txt = sel || prompt('Link text:', url) || url;
        cm.replaceSelection('<a href="' + url + '">' + txt + '</a>');
    }

    function insertImg(cm) {
        var url = prompt('Image URL:', 'https://');
        if (!url) return;
        var alt = prompt('Alt text:', '') || '';
        cm.replaceSelection('<img src="' + url + '" alt="' + alt + '">');
    }

    function insertList(cm, type) {
        var sel   = cm.getSelection();
        var lines = sel ? sel.split('\n').filter(function (l) { return l.trim(); }) : [''];
        var items = lines.map(function (l) { return '  <li>' + l.trim() + '</li>'; }).join('\n');
        var result = '<' + type + '>\n' + items + '\n</' + type + '>';
        if (sel) {
            cm.replaceSelection(result);
        } else {
            var cur = cm.getCursor();
            cm.replaceRange(result, cur);
            cm.setCursor({ line: cur.line + 1, ch: 6 });
        }
    }

    // ── Snippet insert with {|} cursor marker ─────────────────────────────────

    function insertSnippet(cm, code) {
        var marker = '{|}';
        var idx    = code.indexOf(marker);

        if (idx === -1) {
            // No marker — just insert at cursor
            cm.replaceSelection(code);
            return;
        }

        // Split around marker
        var before = code.substring(0, idx);
        var after  = code.substring(idx + marker.length);
        var clean  = before + after;

        // Remember where selection starts
        var from = cm.getCursor('from');
        cm.replaceSelection(clean);

        // Calculate cursor position inside the inserted text
        var beforeLines = before.split('\n');
        var curLine = from.line + beforeLines.length - 1;
        var curCh   = beforeLines.length === 1
            ? from.ch + beforeLines[0].length
            : beforeLines[beforeLines.length - 1].length;

        cm.setCursor({ line: curLine, ch: curCh });
    }

    // ── Save ──────────────────────────────────────────────────────────────────

    $saveBtn.on('click', doSave);

    function doSave() {
        if (!editor) return;
        setStatus('Saving...', 'info');
        $saveBtn.prop('disabled', true);
        $.post(rfheData.ajax_url, {
            action:  'rfhe_save_post',
            post_id: rfheData.post_id,
            nonce:   rfheData.nonce,
            content: editor.getValue(),
        }, function (res) {
            $saveBtn.prop('disabled', false);
            if (!res.success) { setStatus('✗ ' + (res.data || 'Error'), 'error'); return; }
            isDirty = false;
            setStatus('✓ Saved', 'ok');
            exitEditMode(res.data.rendered);
        }).fail(function () {
            $saveBtn.prop('disabled', false);
            setStatus('✗ Network error', 'error');
        });
    }

    // ── Cancel ────────────────────────────────────────────────────────────────

    $cancelBtn.on('click', doCancel);

    function doCancel() {
        if (isDirty && !confirm('Discard unsaved changes?')) return;
        exitEditMode(originalHTML);
        setStatus('', '');
    }

    // ── Exit edit mode ────────────────────────────────────────────────────────

    function exitEditMode(html) {
        if (editor) { editor.toTextArea(); editor = null; }
        if ($target) {
            $target.off('.rfhe');
            $target.removeClass('rfhe-editing').html(html);
            $target = null;
        }
        isDirty = false; isEditing = false;
        hideEditActions();
        $bar.addClass('rfhe-bar-preview').removeClass('rfhe-bar-editing');
    }

    // ── UI helpers ────────────────────────────────────────────────────────────

    function showEditActions() { $editBtn.hide(); $editActions.show(); $pickBtn.hide(); }
    function hideEditActions() { $editBtn.show(); $editActions.hide(); $pickBtn.show(); }
    function setStatus(msg, type) { $status.text(msg).removeClass('ok warn error info').addClass(type); }

    $(document).on('keydown', function (e) {
        if ((e.ctrlKey || e.metaKey) && e.key === 'e' && !isEditing) {
            e.preventDefault();
            $editBtn.trigger('click');
        }
    });

    $(window).on('beforeunload', function () {
        if (isDirty) return 'You have unsaved changes!';
    });

});
