<?php
/*
 * Plugin Name:  Re{code} Front HTML Editor
 * Plugin URI:   https://recodecommerce.com/wordpress-plugins/recode-front-html-editor
 * Description:  Frontend HTML editor for WordPress — edit post_content directly on the page and preview changes instantly without a page reload.
 * Version:      3.5.1
 * Requires at least: 5.9
 * Requires PHP: 7.4
 * Author:       re{code}commerce
 * Author URI:   https://recodecommerce.com
 * License:      GPL v2 or later
 * License URI:  https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:  recode-front-html-editor
 * Domain Path:  /languages
*/

if ( ! defined( 'ABSPATH' ) ) exit;

class Recode_Front_HTML_Editor {

    const VERSION    = '3.5.1';
    const OPTION_KEY = 'rfhe_settings';
    const PLUGIN_URL = 'https://recodecommerce.com/wordpress-plugins/recode-front-html-editor';
    const GITHUB_URL = 'https://github.com/recodecommerce/recode-front-html-editor';

    public function __construct() {
        add_action( 'wp_enqueue_scripts',       [ $this, 'enqueue' ] );
        add_action( 'wp_footer',                [ $this, 'render_toolbar' ] );
        add_action( 'wp_ajax_rfhe_get_content', [ $this, 'ajax_get_content' ] );
        add_action( 'wp_ajax_rfhe_save_post',   [ $this, 'ajax_save_post' ] );
        add_action( 'admin_menu',               [ $this, 'add_settings_page' ] );
        add_action( 'admin_init',               [ $this, 'register_settings' ] );
        add_action( 'admin_enqueue_scripts',    [ $this, 'admin_enqueue' ] );
    }

    // ── Helpers ───────────────────────────────────────────────────────────────

    private function opt( string $key, $default = '' ) {
        $opts = get_option( self::OPTION_KEY, [] );
        return $opts[ $key ] ?? $default;
    }

    private function is_allowed(): bool {
        if ( ! is_singular() ) return false;
        $post_id = get_queried_object_id();
        return $post_id && current_user_can( 'edit_post', $post_id );
    }

    private function get_selector(): string {
        $custom = trim( $this->opt( 'custom_selector', '' ) );
        if ( $custom !== '' ) return $custom;
        return $this->opt( 'default_selector', '.entry-content' ) ?: '.entry-content';
    }

    private function get_theme(): string {
        return $this->opt( 'editor_theme', 'dark' ) === 'light' ? 'light' : 'dark';
    }

    // ── Frontend: enqueue ─────────────────────────────────────────────────────

    public function enqueue() {
        if ( ! $this->is_allowed() ) return;

        $cm = 'https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.16/';

        wp_enqueue_style( 'codemirror',       $cm . 'codemirror.min.css',    [], '5.65.16' );
        wp_enqueue_style( 'codemirror-theme',
            $this->get_theme() === 'light'
                ? $cm . 'theme/eclipse.min.css'
                : $cm . 'theme/dracula.min.css',
            [ 'codemirror' ], '5.65.16'
        );

        wp_enqueue_script( 'codemirror',               $cm . 'codemirror.min.js',                 [],                                                                 '5.65.16', true );
        wp_enqueue_script( 'codemirror-xml',           $cm . 'mode/xml/xml.min.js',               [ 'codemirror' ],                                                   '5.65.16', true );
        wp_enqueue_script( 'codemirror-css-mode',      $cm . 'mode/css/css.min.js',               [ 'codemirror' ],                                                   '5.65.16', true );
        wp_enqueue_script( 'codemirror-js-mode',       $cm . 'mode/javascript/javascript.min.js', [ 'codemirror' ],                                                   '5.65.16', true );
        wp_enqueue_script( 'codemirror-htmlmixed',     $cm . 'mode/htmlmixed/htmlmixed.min.js',   [ 'codemirror-xml', 'codemirror-css-mode', 'codemirror-js-mode' ], '5.65.16', true );
        wp_enqueue_script( 'codemirror-closetag',      $cm . 'addon/edit/closetag.min.js',        [ 'codemirror' ],                                                   '5.65.16', true );
        wp_enqueue_script( 'codemirror-matchbrackets', $cm . 'addon/edit/matchbrackets.min.js',   [ 'codemirror' ],                                                   '5.65.16', true );

        wp_enqueue_script( 'jquery' );

        $js_file = plugin_dir_path( __FILE__ ) . 'assets/editor.js';
        $js_ver  = file_exists( $js_file ) ? filemtime( $js_file ) : self::VERSION;

        wp_enqueue_script(
            'rfhe-js',
            plugin_dir_url( __FILE__ ) . 'assets/editor.js',
            [ 'jquery', 'codemirror-htmlmixed', 'codemirror-closetag', 'codemirror-matchbrackets' ],
            $js_ver,
            true
        );

        $css_file = plugin_dir_path( __FILE__ ) . 'assets/editor.css';
        $css_ver  = file_exists( $css_file ) ? filemtime( $css_file ) : self::VERSION;

        wp_enqueue_style( 'rfhe-css',
            plugin_dir_url( __FILE__ ) . 'assets/editor.css',
            [ 'codemirror' ], $css_ver
        );

        // Build snippets array for JS — only label and code are needed on the frontend.
        $snippets = array_values(
            array_map(
                function ( $s ) {
                    return [
                        'label' => $s['label'],
                        'code'  => $s['code'],
                    ];
                },
                $this->opt( 'snippets', [] )
            )
        );

        wp_localize_script( 'rfhe-js', 'rfheData', [
            'ajax_url'     => admin_url( 'admin-ajax.php' ),
            'post_id'      => get_queried_object_id(),
            'nonce'        => wp_create_nonce( 'rfhe_nonce' ),
            'selector'     => $this->get_selector(),
            'theme'        => $this->get_theme(),
            'settings_url' => admin_url( 'options-general.php?page=rfhe-settings' ),
            'snippets'     => $snippets,
        ] );
    }

    // ── Frontend: toolbar ─────────────────────────────────────────────────────

    public function render_toolbar() {
        if ( ! $this->is_allowed() ) return;
        $settings_url = admin_url( 'options-general.php?page=rfhe-settings' );
        $theme        = $this->get_theme();
        ?>
        <div id="rfhe-bar" class="rfhe-bar-preview rfhe-theme-<?php echo esc_attr( $theme ); ?>">
            <span id="rfhe-brand">&#9999;&#65039; HTML Editor</span>
            <button id="rfhe-edit-btn"><?php esc_html_e( 'Edit', 'recode-front-html-editor' ); ?></button>
            <div id="rfhe-edit-actions" style="display:none">
                <span id="rfhe-status"></span>
                <button id="rfhe-save-btn">&#128190; <?php esc_html_e( 'Save &amp; Preview', 'recode-front-html-editor' ); ?></button>
                <button id="rfhe-cancel-btn">&#10005; <?php esc_html_e( 'Cancel', 'recode-front-html-editor' ); ?></button>
            </div>
            <a id="rfhe-settings-link"
               href="<?php echo esc_url( $settings_url ); ?>"
               title="<?php esc_attr_e( 'Settings', 'recode-front-html-editor' ); ?>"
               target="_blank" rel="noopener noreferrer">&#9881;</a>
            <button id="rfhe-pick-btn" title="<?php esc_attr_e( 'Click to pick a content element and get its CSS selector', 'recode-front-html-editor' ); ?>">&#127919; <?php esc_html_e( 'Pick element', 'recode-front-html-editor' ); ?></button>
        </div>
        <?php
    }

    // ── AJAX ──────────────────────────────────────────────────────────────────

    public function ajax_get_content() {
        check_ajax_referer( 'rfhe_nonce', 'nonce' );

        $post_id = intval( $_POST['post_id'] ?? 0 );

        if ( ! $post_id || ! current_user_can( 'edit_post', $post_id ) ) {
            wp_send_json_error( 'Access denied' );
        }

        $post = get_post( $post_id );
        if ( ! $post ) {
            wp_send_json_error( 'Post not found' );
        }

        wp_send_json_success( [ 'content' => $post->post_content ] );
    }

    public function ajax_save_post() {
        check_ajax_referer( 'rfhe_nonce', 'nonce' );

        $post_id = intval( $_POST['post_id'] ?? 0 );

        if ( ! $post_id || ! current_user_can( 'edit_post', $post_id ) ) {
            wp_send_json_error( 'Access denied' );
        }

        $content = wp_kses_post( wp_unslash( $_POST['content'] ?? '' ) );
        $result  = wp_update_post( [ 'ID' => $post_id, 'post_content' => $content ], true );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_message() );
        }

        $post     = get_post( $post_id );
        $rendered = apply_filters( 'the_content', $post->post_content ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound        

        wp_send_json_success( [ 'rendered' => $rendered ] );
    }

    // ── Settings ──────────────────────────────────────────────────────────────

    private function default_selectors(): array {
        return [
            '.entry-content'         => '.entry-content — Twenty* / Astra / GeneratePress / Kadence',
            '.post-content'          => '.post-content — Divi / OceanWP / Flatsome',
            '.wp-block-post-content' => '.wp-block-post-content — Block & FSE themes',
            'article .content'       => 'article .content — various',
            'main article'           => 'main article — generic fallback',
        ];
    }

    public function add_settings_page() {
        add_options_page(
            __( 'Front HTML Editor — Settings', 'recode-front-html-editor' ),
            __( 'Front HTML Editor', 'recode-front-html-editor' ),
            'manage_options',
            'rfhe-settings',
            [ $this, 'render_settings_page' ]
        );
    }

    public function register_settings() {
        register_setting(
            self::OPTION_KEY,
            self::OPTION_KEY,
            [ 'sanitize_callback' => [ $this, 'sanitize_options' ] ]
        );

        // Section: Content Selector
        add_settings_section( 'rfhe_sel', __( 'Content Selector', 'recode-front-html-editor' ), function () {
            echo '<p class="description">'
                . esc_html__( 'Defines which DOM element contains the editable post content. If a Custom selector is provided it takes priority over the Default one.', 'recode-front-html-editor' )
                . '</p>';
        }, 'rfhe-settings' );

        add_settings_field( 'rfhe_default_selector', __( 'Default selector', 'recode-front-html-editor' ),
            [ $this, 'field_default_selector' ], 'rfhe-settings', 'rfhe_sel' );
        add_settings_field( 'rfhe_custom_selector',  __( 'Custom selector', 'recode-front-html-editor' ),
            [ $this, 'field_custom_selector' ],  'rfhe-settings', 'rfhe_sel' );

        // Section: Editor Appearance
        add_settings_section( 'rfhe_app', __( 'Editor Appearance', 'recode-front-html-editor' ), function () {
            echo '<p class="description">'
                . esc_html__( 'Visual theme applied to the CodeMirror code editor.', 'recode-front-html-editor' )
                . '</p>';
        }, 'rfhe-settings' );

        add_settings_field( 'rfhe_editor_theme', __( 'Color theme', 'recode-front-html-editor' ),
            [ $this, 'field_editor_theme' ], 'rfhe-settings', 'rfhe_app' );

        // Section: Custom Snippets
        add_settings_section( 'rfhe_snip', __( 'Custom Snippets', 'recode-front-html-editor' ), function () {
            echo '<p class="description">'
                . esc_html__( 'Add reusable HTML blocks to the editor toolbar. Use {|} in the code to mark where the cursor should be placed after inserting.', 'recode-front-html-editor' )
                . '</p>';
        }, 'rfhe-settings' );

        add_settings_field( 'rfhe_snippets', __( 'Snippets', 'recode-front-html-editor' ),
            [ $this, 'field_snippets' ], 'rfhe-settings', 'rfhe_snip' );
    }

    public function sanitize_options( array $input ): array {
        $clean = [];

        $clean['default_selector'] = in_array( $input['default_selector'] ?? '', array_keys( $this->default_selectors() ), true )
            ? $input['default_selector'] : '.entry-content';
        $clean['custom_selector']  = sanitize_text_field( $input['custom_selector'] ?? '' );
        $clean['editor_theme']     = in_array( $input['editor_theme'] ?? '', [ 'dark', 'light' ], true )
            ? $input['editor_theme'] : 'dark';

        // Snippets: keep only rows with a non-empty label
        $clean['snippets'] = [];
        foreach ( (array) ( $input['snippets'] ?? [] ) as $row ) {
            $label = sanitize_text_field( $row['label'] ?? '' );
            $code  = wp_kses_post( wp_unslash( $row['code'] ?? '' ) );
            if ( $label !== '' ) {
                $clean['snippets'][] = [ 'label' => $label, 'code' => $code ];
            }
        }

        return $clean;
    }

    public function field_default_selector() {
        $current = $this->opt( 'default_selector', '.entry-content' );
        $name    = self::OPTION_KEY . '[default_selector]';
        echo '<select name="' . esc_attr( $name ) . '" class="regular-text">';
        foreach ( $this->default_selectors() as $value => $label ) {
            printf(
                '<option value="%s"%s>%s</option>',
                esc_attr( $value ),
                selected( $current, $value, false ),
                esc_html( $label )
            );
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__( 'Standard selectors covering the most popular WordPress themes.', 'recode-front-html-editor' ) . '</p>';
    }

    public function field_custom_selector() {
        $val = $this->opt( 'custom_selector', '' );
        printf(
            '<input type="text" name="%s" value="%s" class="regular-text" placeholder="#g_content  or  .my-theme-wrapper">',
            esc_attr( self::OPTION_KEY . '[custom_selector]' ),
            esc_attr( $val )
        );
        echo '<p class="description">' . esc_html__( 'Any valid CSS selector. Overrides the Default selector. Leave blank to use the Default.', 'recode-front-html-editor' ) . '</p>';
    }

    public function field_editor_theme() {
        $current = $this->opt( 'editor_theme', 'dark' );
        $name    = self::OPTION_KEY . '[editor_theme]';
        $options = [
            'dark'  => '&#127769; ' . esc_html__( 'Dark (Dracula)', 'recode-front-html-editor' ),
            'light' => '&#9728;&#65039; ' . esc_html__( 'Light (Eclipse)', 'recode-front-html-editor' ),
        ];
        foreach ( $options as $val => $label ) {
            printf(
                '<label style="margin-right:24px;cursor:pointer"><input type="radio" name="%s" value="%s"%s> %s</label>',
                esc_attr( $name ),
                esc_attr( $val ),
                checked( $current, $val, false ),
                wp_kses( $label, [] )
        );
        }
    }

    public function field_snippets() {
        $snippets  = $this->opt( 'snippets', [] );
        $base_name = self::OPTION_KEY . '[snippets]';
        ?>
        <div id="rfhe-snippets-list">
        <?php if ( empty( $snippets ) ) : ?>
            <p class="rfhe-no-snippets description"><?php esc_html_e( 'No snippets yet. Click "Add snippet" below.', 'recode-front-html-editor' ); ?></p>
        <?php endif; ?>
        <?php foreach ( $snippets as $i => $snippet ) : ?>
            <div class="rfhe-snippet-row">
                <div class="rfhe-snippet-header">
                    <input
                        type="text"
                        name="<?php echo esc_attr( $base_name . '[' . $i . '][label]' ); ?>"
                        value="<?php echo esc_attr( $snippet['label'] ); ?>"
                        placeholder="<?php esc_attr_e( 'Button label (e.g. See Also)', 'recode-front-html-editor' ); ?>"
                        class="regular-text rfhe-snippet-label"
                    >
                    <button type="button" class="button rfhe-snippet-remove">&#10005; <?php esc_html_e( 'Remove', 'recode-front-html-editor' ); ?></button>
                </div>
                <textarea
                    name="<?php echo esc_attr( $base_name . '[' . $i . '][code]' ); ?>"
                    rows="4"
                    class="large-text code rfhe-snippet-code"
                    placeholder="<?php esc_attr_e( 'HTML code. Use {|} to mark cursor position.', 'recode-front-html-editor' ); ?>"
                ><?php echo esc_textarea( $snippet['code'] ); ?></textarea>
                <p class="description">
                    <?php esc_html_e( 'Tip: place {|} where you want the cursor after inserting.', 'recode-front-html-editor' ); ?>
                </p>
            </div>
        <?php endforeach; ?>
        </div>

        <button type="button" id="rfhe-add-snippet" class="button button-secondary" style="margin-top:10px;">
            + <?php esc_html_e( 'Add snippet', 'recode-front-html-editor' ); ?>
        </button>

        <script>
        (function () {
            var list     = document.getElementById('rfhe-snippets-list');
            var addBtn   = document.getElementById('rfhe-add-snippet');
            var baseName = <?php echo wp_json_encode( $base_name ); ?>;

            function reindex() {
                var rows = list.querySelectorAll('.rfhe-snippet-row');
                rows.forEach(function (row, i) {
                    row.querySelectorAll('[name]').forEach(function (el) {
                        el.name = el.name.replace(/\[\d+\]/, '[' + i + ']');
                    });
                });
                // Show/hide placeholder
                var ph = list.querySelector('.rfhe-no-snippets');
                if (ph) ph.style.display = rows.length ? 'none' : '';
            }

            function bindRemove(row) {
                row.querySelector('.rfhe-snippet-remove').addEventListener('click', function () {
                    row.remove();
                    reindex();
                });
            }

            // Bind existing rows
            list.querySelectorAll('.rfhe-snippet-row').forEach(bindRemove);

            addBtn.addEventListener('click', function () {
                var idx = list.querySelectorAll('.rfhe-snippet-row').length;
                var row = document.createElement('div');
                row.className = 'rfhe-snippet-row';
                row.innerHTML =
                    '<div class="rfhe-snippet-header">' +
                        '<input type="text" name="' + baseName + '[' + idx + '][label]" value="" placeholder="<?php echo esc_js( __( 'Button label (e.g. See Also)', 'recode-front-html-editor' ) ); ?>" class="regular-text rfhe-snippet-label">' +
                        '<button type="button" class="button rfhe-snippet-remove">&#10005; <?php echo esc_js( __( 'Remove', 'recode-front-html-editor' ) ); ?></button>' +
                    '</div>' +
                    '<textarea name="' + baseName + '[' + idx + '][code]" rows="4" class="large-text code rfhe-snippet-code" placeholder="<?php echo esc_js( __( 'HTML code. Use {|} to mark cursor position.', 'recode-front-html-editor' ) ); ?>"></textarea>' +
                    '<p class="description"><?php echo esc_js( __( 'Tip: place {|} where you want the cursor after inserting.', 'recode-front-html-editor' ) ); ?></p>';
                list.appendChild(row);
                bindRemove(row);
                reindex();
                row.querySelector('.rfhe-snippet-label').focus();
            });
        })();
        </script>
        <?php
    }

    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        ?>
        <div class="wrap rfhe-wrap">
            <h1>&#9999;&#65039; Re{code} Front HTML Editor <span class="rfhe-ver">v<?php echo esc_html( self::VERSION ); ?></span></h1>
            <?php settings_errors( self::OPTION_KEY ); ?>
            <div class="rfhe-layout">
                <div class="rfhe-main card">
                    <form method="post" action="options.php">
                        <?php
                        settings_fields( self::OPTION_KEY );
                        do_settings_sections( 'rfhe-settings' );
                        submit_button( __( 'Save Settings', 'recode-front-html-editor' ) );
                        ?>
                    </form>
                </div>
                <div class="rfhe-sidebar">
                    <div class="rfhe-card">
                        <h3>&#9000;&#65039; <?php esc_html_e( 'Keyboard Shortcuts', 'recode-front-html-editor' ); ?></h3>
                        <table class="rfhe-kb">
                            <tr><td><kbd>Ctrl+E</kbd></td><td><?php esc_html_e( 'Open / close editor', 'recode-front-html-editor' ); ?></td></tr>
                            <tr><td><kbd>Ctrl+S</kbd></td><td><?php esc_html_e( 'Save &amp; Preview', 'recode-front-html-editor' ); ?></td></tr>
                            <tr><td><kbd>Escape</kbd></td><td><?php esc_html_e( 'Cancel', 'recode-front-html-editor' ); ?></td></tr>
                            <tr><td><kbd>Ctrl+B</kbd></td><td><?php esc_html_e( 'Bold', 'recode-front-html-editor' ); ?></td></tr>
                            <tr><td><kbd>Ctrl+I</kbd></td><td><?php esc_html_e( 'Italic', 'recode-front-html-editor' ); ?></td></tr>
                            <tr><td><kbd>Ctrl+K</kbd></td><td><?php esc_html_e( 'Insert link', 'recode-front-html-editor' ); ?></td></tr>
                        </table>
                    </div>
                    <div class="rfhe-card">
                        <h3>&#128269; <?php esc_html_e( 'How the Selector Works', 'recode-front-html-editor' ); ?></h3>
                        <ol>
                            <li><?php esc_html_e( 'If Custom selector is set → it is used', 'recode-front-html-editor' ); ?></li>
                            <li><?php esc_html_e( 'Otherwise Default selector is used', 'recode-front-html-editor' ); ?></li>
                            <li><?php esc_html_e( 'If nothing matches → warning shown in toolbar', 'recode-front-html-editor' ); ?></li>
                        </ol>
                        <p><?php esc_html_e( 'Use browser DevTools → Inspector to find the right element.', 'recode-front-html-editor' ); ?></p>
                        <p><?php esc_html_e( 'Examples:', 'recode-front-html-editor' ); ?> <code>#g_content</code> &middot; <code>.post-body</code> &middot; <code>article .text</code></p>
                    </div>
                    <div class="rfhe-card">
                        <h3>&#127981; <?php esc_html_e( 'Snippet Tips', 'recode-front-html-editor' ); ?></h3>
                        <p><?php esc_html_e( 'Use {|} to set cursor position after insert.', 'recode-front-html-editor' ); ?></p>
                        <p><?php esc_html_e( 'Example:', 'recode-front-html-editor' ); ?></p>
                        <pre class="rfhe-snippet-example">&lt;div class="note"&gt;
  {|}
&lt;/div&gt;</pre>
                    </div>
                    <div class="rfhe-card rfhe-card-about">
                        <h3>&#8505;&#65039; <?php esc_html_e( 'About', 'recode-front-html-editor' ); ?></h3>
                        <p><?php esc_html_e( 'Edits post_content directly in the DB via AJAX. No page reload. Only visible to users with edit_post capability for the current post.', 'recode-front-html-editor' ); ?></p>
                        <p><a href="<?php echo esc_url( self::PLUGIN_URL ); ?>" target="_blank" rel="noopener noreferrer">re{code}commerce &rarr;</a></p>
                    </div>
                </div>
            </div>

            <div style="margin-top:40px;padding-top:20px;border-top:1px solid #ddd;text-align:center;">
                <p style="font-size:14px;color:#666;margin-bottom:10px;">
                    <?php
                    printf(
                        '<strong>%s</strong> %s <a href="https://recodecommerce.com" target="_blank" style="color:#e94560;text-decoration:none;font-weight:600;">re{code}commerce</a>',
                        esc_html__( 'Re{code} Front HTML Editor', 'recode-front-html-editor' ),
                        esc_html__( 'is developed by', 'recode-front-html-editor' )
                    );
                    ?>
                </p>
                <p style="font-size:13px;color:#999;">
                    <?php
                    printf(
                        '<a href="%s" target="_blank" style="color:#0073aa;text-decoration:none;">%s</a> | ',
                        esc_url( self::PLUGIN_URL ),
                        esc_html__( 'Plugin Page', 'recode-front-html-editor' )
                    );
                    printf(
                        '<a href="%s" target="_blank" style="color:#0073aa;text-decoration:none;">%s</a> | ',
                        esc_url( self::GITHUB_URL ),
                        esc_html__( 'GitHub', 'recode-front-html-editor' )
                    );
                    printf(
                        '<a href="https://recodecommerce.com/#contact" target="_blank" style="color:#0073aa;text-decoration:none;">%s</a>',
                        esc_html__( 'Need Custom Development?', 'recode-front-html-editor' )
                    );
                    ?>
                </p>
            </div>
        </div>
        <?php
    }

    public function admin_enqueue( string $hook ) {
        if ( $hook !== 'settings_page_rfhe-settings' ) return;
        wp_add_inline_style( 'wp-admin', '
            .card { margin-top: 0px;}    
            .rfhe-wrap h1{display:flex;align-items:center;gap:10px;font-size:22px}
            .rfhe-ver{font-size:12px;font-weight:400;background:#f0f0f1;color:#777;padding:2px 9px;border-radius:20px}
            .rfhe-layout{display:flex;gap:24px;align-items:flex-start;margin-top:20px}
            .rfhe-main{flex:1;min-width:0;padding:24px 28px}
            .rfhe-sidebar{width:300px;flex-shrink:0;display:flex;flex-direction:column;gap:16px}
            .rfhe-card{background:#fff;border:1px solid #c3c4c7;border-radius:3px;padding:16px 20px;box-shadow:0 1px 1px rgba(0,0,0,.04)}
            .rfhe-card h3{margin:0 0 12px;font-size:13px;text-transform:uppercase;letter-spacing:.05em;color:#3c434a}
            .rfhe-card-about{border-left:4px solid #2271b1}
            .rfhe-card ol{margin:0 0 10px;padding-left:18px}
            .rfhe-card li,.rfhe-card p{font-size:13px;margin:6px 0}
            .rfhe-card code{background:#f0f0f1;padding:1px 5px;border-radius:3px;font-size:12px}
            .rfhe-kb{border-collapse:collapse;font-size:13px;width:100%}
            .rfhe-kb td{padding:4px 8px 4px 0;vertical-align:middle}
            .rfhe-kb td:first-child{width:90px}
            kbd{background:#f0f0f1;border:1px solid #ccc;border-radius:3px;padding:1px 7px;font-size:12px;font-family:monospace}
            .rfhe-snippet-row{background:#f9f9f9;border:1px solid #ddd;border-radius:4px;padding:14px 16px;margin-bottom:12px}
            .rfhe-snippet-header{display:flex;align-items:center;gap:10px;margin-bottom:8px}
            .rfhe-snippet-label{flex:1}
            .rfhe-snippet-code{font-family:monospace;font-size:12px;resize:vertical}
            .rfhe-snippet-example{background:#f0f0f1;border:1px solid #ddd;border-radius:3px;padding:8px 12px;font-size:12px;margin:4px 0 0}
            .rfhe-main .form-table th,.rfhe-main .form-table td{display:block;width:100%;padding-left:0;box-sizing:border-box;}
            .rfhe-main .form-table tr{display:block;margin-bottom:20px;}
            .rfhe-main .form-table th{padding-bottom:5px;padding-top:0;font-weight:600;}
            .rfhe-main .form-table td{padding-top:0;}
            @media(max-width:900px){.rfhe-layout{flex-direction:column}.rfhe-sidebar{width:100%}}
        ' );
    }
}

new Recode_Front_HTML_Editor();
