<?php
/**
 * Plugin Name: Recode Cron Viewer
 * Plugin URI: https://recodecommerce.com/wordpress-plugins/recode-cron-viewer
 * Description: Displays all scheduled WP-Cron tasks in the admin panel.
 * Version: 1.1.0
 * Author: re{code}commerce
 * Author URI: https://recodecommerce.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: recode-cron-viewer
 * Text Domain: recode-cron-viewer
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class RECODE_Cron_Viewer {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
    }

    public function add_menu() {
        add_menu_page(
            esc_html__( 'Cron Tasks', 'recode-cron-viewer' ),           // Page title
            esc_html__( 'Cron Viewer', 'recode-cron-viewer' ),          // Menu title
            'manage_options',       // Capability
            'recode-cron-viewer',       // Menu slug
            array( $this, 'render_page' ), // Callback
            'dashicons-clock',      // Icon
            100                     // Position
        );
    }

    public function render_page() {
        // Retrieve all cron events
        $crons = _get_cron_array();
        $schedules = wp_get_schedules();
        $date_format = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );

        echo '<div class="wrap">';
        echo '<h1>' . esc_html__( 'WP-Cron Tasks', 'recode-cron-viewer' ) . '</h1>';
        
        if ( empty( $crons ) ) {
            echo '<div class="notice notice-warning"><p>' . esc_html__( 'No scheduled tasks found.', 'recode-cron-viewer' ) . '</p></div>';
            echo '</div>';
            return;
        }

        echo '<table class="widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th style="width: 20%;">' . esc_html__( 'Next Run', 'recode-cron-viewer' ) . '</th>';
        echo '<th style="width: 25%;">' . esc_html__( 'Hook', 'recode-cron-viewer' ) . '</th>';
        echo '<th>' . esc_html__( 'Arguments', 'recode-cron-viewer' ) . '</th>';
        echo '<th style="width: 15%;">' . esc_html__( 'Schedule', 'recode-cron-viewer' ) . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';

        foreach ( $crons as $timestamp => $cronhooks ) {
            foreach ( $cronhooks as $hook => $keys ) {
                foreach ( $keys as $k => $v ) {
                    
                    $schedule_label = esc_html__( 'One-off', 'recode-cron-viewer' );
                    if ( isset( $v['schedule'] ) && $v['schedule'] ) {
                        $schedule_label = isset($schedules[$v['schedule']]) ? $schedules[$v['schedule']]['display'] : $v['schedule'];
                    }

                    $time_diff = human_time_diff( time(), $timestamp );
                    $is_past = $timestamp < time();
                    
                    // Adjust time to local WP time
                    $local_time = get_date_from_gmt( date( 'Y-m-d H:i:s', $timestamp ), $date_format );

                    echo '<tr>';
                    
                    // Time column
                    echo '<td>';
                    echo '<strong>' . esc_html( $local_time ) . '</strong>';
                    echo '<br>';
                    if ( $is_past ) {
                        printf(
                            '<span style="color: #d63638;">(%s)</span>',
                            sprintf( esc_html__( 'overdue by %s', 'recode-cron-viewer' ), esc_html( $time_diff ) )
                        );
                    } else {
                        printf(
                            '<span style="color: #00a32a;">(%s)</span>',
                            sprintf( esc_html__( 'in %s', 'recode-cron-viewer' ), esc_html( $time_diff ) )
                        );
                    }
                    echo '</td>';

                    // Hook column
                    echo '<td><strong>' . esc_html( $hook ) . '</strong></td>';

                    // Args column
                    echo '<td>';
                    if ( ! empty( $v['args'] ) ) {
                        echo '<pre style="margin: 0; white-space: pre-wrap;">' . esc_html( print_r( $v['args'], true ) ) . '</pre>';
                    } else {
                        echo '<span style="color: #aaa;">' . esc_html__( 'No arguments', 'recode-cron-viewer' ) . '</span>';
                    }
                    echo '</td>';

                    // Schedule column
                    echo '<td>' . esc_html( $schedule_label ) . '</td>';
                    echo '</tr>';
                }
            }
        }

		echo '</tbody>';
		echo '</table>';

		echo '<div style="margin-top: 40px; padding-top: 20px; border-top: 1px solid #ddd; text-align: center;">';
		echo '<p style="font-size: 14px; color: #666; margin-bottom: 10px;">';
		printf(
			'<strong>%s</strong> %s <a href="https://recodecommerce.com" target="_blank" style="color: #e94560; text-decoration: none; font-weight: 600;">re{code}commerce</a>',
			esc_html__( 'Re{code} Cron Viewer', 'recode-cron-viewer' ),
			esc_html__( 'is developed by', 'recode-cron-viewer' )
		);
		echo '</p>';

		echo '<p style="font-size: 13px; color: #999;">';
		printf(
			'<a href="https://recodecommerce.com/wordpress-plugins/recode-cron-viewer/" target="_blank" style="color: #0073aa; text-decoration: none;">%s</a> | ',
			esc_html__( 'Plugin Page', 'recode-cron-viewer' )
		);
		printf(
			'<a href="https://github.com/recodecommerce/recode-cron-viewer" target="_blank" style="color: #0073aa; text-decoration: none;">%s</a> | ',
			esc_html__( 'GitHub', 'recode-cron-viewer' )
		);
		printf(
			'<a href="https://recodecommerce.com/#contact" target="_blank" style="color: #0073aa; text-decoration: none;">%s</a>',
			esc_html__( 'Need Custom Development?', 'recode-cron-viewer' )
		);
		echo '</p>';
		echo '</div>';

		echo '</div>';
    }
}

new RECODE_Cron_Viewer();
